unit MapUpdater;

interface

uses
  Windows, SysUtils, Classes, Graphics, Controls, Generics.Collections, SyncObjs,
  GraphicEx, StdCtrls, Dialogs, ExtCtrls;

type
  PBitmap = ^TBitmap;
  TLine = array[0..MaxInt div SizeOf(TRGBQUAD) - 1] of TRGBQUAD;
  PLine = ^TLine;

  TMapState = (msNew, msExisting, msDeleted);

  PMapInfo = ^TMapInfo;
  TMapInfo = record
    FileName: String;
    MapName: String;
    MapDesc: String;
    MapAuthor: String;
    FileSizePretty: String;
    FileSize: Integer;
    PlayerCount: Integer;
    State: TMapState;
    Bitmap: TBitmap;
  end;

  TMapUpdater = class(TThread)
  private
  protected
    procedure Execute; override;
  public
    Queue: TList<PMapInfo>;
    QueueLock: TCriticalSection;
    constructor Create(CreateSuspended: Boolean);
    destructor Destroy; override;
    procedure Process(MapInfo: PMapInfo);
    procedure UpdateInterface;
  end;

implementation

uses MapSelector, MpqAPI, GameSetup, StreamUtils;

function ResampleSubBitmap(Bitmap: TBitmap; var Rect: TRect; XPos, YPos, Width, Height: Integer): TRGBQuad; inline;
var
  r, g, b: Cardinal;
  Line: PLine;
  x, y, z: Integer;
begin
  z := (Width * Height);
  r := 0;
  g := 0;
  b := 0;

  if (YPos + Height) >= (Rect.Bottom - Rect.Top) then Height := ((Rect.Bottom - Rect.Top) - YPos) - 1;
  if (XPos + Width) >= (Rect.Right - Rect.Left) then Width := ((Rect.Right - Rect.Left) - XPos) - 1;

  for y := YPos to YPos + Height do
  begin
    Line := Bitmap.ScanLine[Rect.Top + y];
    for x := XPos to XPos + Width do
    begin
      r := r + Line[Rect.left + x].rgbRed;
      g := g + Line[Rect.left + x].rgbGreen;
      b := b + Line[Rect.left + x].rgbBlue;
      Inc(z);
    end;
  end;

  if (z = 0) then z := 1;

  r := Round((r / z) * 1.4);
  if (r > 255) then r := 255;
  g := Round((g / z) * 1.4);
  if (g > 255) then g := 255;
  b := Round((b / z) * 1.4);
  if (b > 255) then b := 255;

  Result.rgbRed   := r;
  Result.rgbGreen := g;
  Result.rgbBlue  := b;
end;

function ResampleBitmap(Bitmap: TBitmap; var SourceRect: TRect; NewWidth, NewHeight: Integer): TBitmap;
var
  Temp: TBitmap;
  Line: PLine;
  x, y: Integer;
  Blockheight, Blockwidth: Cardinal;
  BlockPosX, BlockPosY: Single;
  BlockDiffX, BlockDiffY: Single;
  XPos, YPos: Single;
  DiffX, Diffy: Single;
begin
  Temp := TBitmap.Create;
  Temp.PixelFormat := pf32Bit;

  Temp.Height := NewHeight + 2;
  Temp.Width := NewWidth + 2;

  Temp.Canvas.Brush.Color := clGrayText;
  Temp.Canvas.FillRect(Rect(0, 0, NewWidth + 2, NewHeight + 2));

  BlockDiffY := ((SourceRect.Bottom - SourceRect.Top) / NewHeight);
  BlockDiffX := ((SourceRect.Right - SourceRect.Left) / NewWidth);

  BlockHeight := Trunc(BlockDiffY);
  BlockWidth  := Trunc(BlockDiffY);

  DiffX := 1;
  DiffY := 1;

  BlockPosY := 0;
  YPos      := 0;

  for y := 0 to NewHeight - 1 do
  begin
    BlockPosX := 0;
    XPos      := 0;

    Line := Temp.ScanLine[1 + Trunc(YPos)];
    for x := 0 to NewWidth - 1 do
    begin
      Line[1 + Trunc(XPos)] := ResampleSubBitmap(Bitmap, SourceRect,
        Round(BlockPosX), Round(BlockPosY), Blockwidth, BlockHeight);

      BlockPosX := BlockPosX + BlockDiffX;
      XPos      := XPos + DiffX;
    end;

    BlockPosY := BlockPosY + BlockDiffY;
    YPos      := YPos + DiffY;
  end;

  Result := Temp;
end;

function IsQuadBlack(var Quad: TRGBQUAD): Boolean; inline;
begin
  Result := (Quad.rgbRed = 0) and (Quad.rgbGreen = 0) and (Quad.rgbBlue = 0);
end;

function IsHorizontalLineBlack(Bitmap: TBitmap; Y: Integer): Boolean; inline;
var x: Integer;
  Line: PLine;
begin
  Line := Bitmap.ScanLine[Y];

  for x := 0 to Bitmap.Width - 1 do
    if not IsQuadBlack(Line[x]) then
      Exit(False);

  Result := True;
end;


function IsVerticalLineBlack(Bitmap: TBitmap; X: Integer): Boolean; inline;
var y: Integer;
begin
  for y := 0 to Bitmap.Height - 1 do
    if not IsQuadBlack(PLine(Bitmap.ScanLine[y])[X]) then
      Exit(False);

  Result := True;
end;

function FindTopEdge(Bitmap: TBitmap): Integer; inline;
var y: Integer;
begin
  for y := 0 to Bitmap.Height - 1 do
    if not IsHorizontalLineBlack(Bitmap, y) then
      if y - 1 < 0 then
        Exit(0)
      else
        Exit(y - 1);

  Result := Bitmap.Height - 1;
end;

function FindBottomEdge(Bitmap: TBitmap): Integer; inline;
var y: Integer;
begin
  for y := Bitmap.Height - 1 downto 0 do
    if not IsHorizontalLineBlack(Bitmap, y) then
      if y + 1 = Bitmap.Height then
        Exit(Bitmap.Height - 1)
      else
        Exit(y + 1);

  Result := 0;
end;

function FindLeftEdge(Bitmap: TBitmap): Integer; inline;
var x: Integer;
begin
  for x := 0 to Bitmap.Width - 1 do
    if not IsVerticalLineBlack(Bitmap, x) then
      if x - 1 < 0 then
        Exit(0)
      else
        Exit(x - 1);

  Result := Bitmap.Width - 1;
end;

function FindRightEdge(Bitmap: TBitmap): Integer; inline;
var x: Integer;
begin
  for x := Bitmap.Width - 1 downto 0 do
    if not IsVerticalLineBlack(Bitmap, x) then
      if x + 1 = Bitmap.Width then
        Exit(Bitmap.Width - 1)
      else
        Exit(x + 1);

  Result := 0;
end;

procedure FindEdges(Bitmap: TBitmap; var Rect: TRect);
begin
  Rect.Left := FindLeftEdge(Bitmap);
  Rect.Right := FindRightEdge(Bitmap);
  Rect.Top := FindTopEdge(Bitmap);
  Rect.Bottom := FindBottomEdge(Bitmap);
end;

constructor TMapUpdater.Create(CreateSuspended: Boolean);
begin
  inherited Create(CreateSuspended);

  QueueLock := TCriticalSection.Create;
  Queue := TList<PMapInfo>.Create;
end;

destructor TMapUpdater.Destroy;
begin
  QueueLock.Free;
  Queue.Free;

  inherited;
end;

procedure TMapUpdater.Process(MapInfo: PMapInfo);
var
  MapHandle: THandle;
  Picture: TTargaGraphic;
  MemoryStream: TMemoryStream;
  Bitmap: TBitmap;
  NewHeight, NewWidth: Integer;
  Rect: TRect;
  PlayerCount, Null: Byte;
  Strings: TStringList;
begin
  if SFileOpenArchive(PAnsiChar(AnsiString(SearchPath + 'Maps\' + MapInfo.FileName)), 0, MOAU_OPEN_EXISTING, @MapHandle) then
    begin
      try
        MemoryStream := MpqReadFile('enUS.SC2Data\LocalizedData\GameStrings.txt');
        MemoryStream.Position := MemoryStream.Size;
        Null := 0;
        MemoryStream.Write(Null, 1);
        try
          Strings := TStringList.Create;
          try
          Strings.Text := String(UTF8String(RawByteString(PAnsiChar(MemoryStream.Memory))));
          Strings.NameValueSeparator := '=';
          MapInfo.MapAuthor := Strings.Values['DocInfo/Author'];
          MapInfo.MapName := Strings.Values['DocInfo/Name'];
          MapInfo.MapDesc := Strings.Values['DocInfo/DescLong'];
          finally
            Strings.Free;
          end;
        finally
          MemoryStream.Free;
        end;

        MemoryStream := MpqReadFile('MapInfo');
        try
          MemoryStream.Position := 24; // Skip fields
          SkipNullString(MemoryStream); // Tileset
          SkipNullString(MemoryStream); // Title
          MemoryStream.Seek(24, soFromCurrent); // Skip fields
          SkipNullString(MemoryStream); // Loading image
          MemoryStream.Seek(32, soFromCurrent); // Skip fields
          MemoryStream.Read(PlayerCount, 1);

          MapInfo.PlayerCount := PlayerCount - 1;
        finally
          MemoryStream.Free;
        end;

        MemoryStream := MpqReadFile('Minimap.tga');
        try
          Picture := TTargaGraphic.Create;
          try
            Picture.LoadFromStream(MemoryStream);

            Bitmap := TBitmap.Create;
            try
              Bitmap.PixelFormat := pf32Bit;
              Bitmap.Width := Picture.Width;
              Bitmap.Height := Picture.Height;
              Bitmap.Canvas.Draw(0, 0, Picture);

              FindEdges(Bitmap, Rect);

              NewHeight := (64 * Picture.Height) div Picture.Width;

              if NewHeight > 64 then
                begin
                  NewWidth := (64 * Picture.Width) div Picture.Height;
                  NewHeight := 64;
                end
              else
                NewWidth := 64;

              MapInfo.Bitmap := ResampleBitmap(Bitmap, Rect, NewWidth, NewHeight);
            finally
              Bitmap.Free;
            end;

          finally
            Picture.Free;
          end;
        finally
          MemoryStream.Free;
        end;
      finally
        SFileCloseArchive(MapHandle);
      end;
    end;
end;

procedure TMapUpdater.UpdateInterface;
begin
  MapForm.MapList.Repaint;
  GameForm.SelectMap(GameForm.SelectedMap);
end;

procedure TMapUpdater.Execute;
var
  i: Integer;
begin
  while not Terminated do
    begin
      try
        QueueLock.Enter;

        if Queue.Count > 0 then
          begin

            for i := 0 to Queue.Count - 1 do
              Process(Queue[i]);

            Synchronize(UpdateInterface);
          end;

        Queue.Clear;
      finally
        QueueLock.Leave;
      end;

      Sleep(100);
    end;
end;

end.
