unit MapSelector;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, VirtualTrees, StdCtrls, MapUpdater, ExtCtrls;

type
  TMapForm = class(TForm)
    MapList: TVirtualStringTree;
    ButtonSelect: TButton;
    Button2: TButton;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure MapListCompareNodes(Sender: TBaseVirtualTree; Node1,
      Node2: PVirtualNode; Column: TColumnIndex; var Result: Integer);
    procedure MapListFocusChanged(Sender: TBaseVirtualTree; Node: PVirtualNode;
      Column: TColumnIndex);
    procedure MapListFreeNode(Sender: TBaseVirtualTree; Node: PVirtualNode);
    procedure MapListHeaderClick(Sender: TVTHeader; HitInfo: TVTHeaderHitInfo);
    procedure ButtonSelectClick(Sender: TObject);
    procedure MapListDblClick(Sender: TObject);
    procedure MapListGetText(Sender: TBaseVirtualTree; Node: PVirtualNode;
      Column: TColumnIndex; TextType: TVSTTextType; var CellText: string);
    procedure MapListAfterCellPaint(Sender: TBaseVirtualTree;
      TargetCanvas: TCanvas; Node: PVirtualNode; Column: TColumnIndex;
      CellRect: TRect);
  private
    { Private declarations }
  public
    Updater: TMapUpdater;
    SelectedNode: PVirtualNode;
    procedure FlagMaps;
    procedure ClearMaps;
    procedure RefreshFolder;
    function GetMap(Name: String; FileSize: Integer): PMapInfo;
    function SelectMap: PMapInfo;
  end;

var
  MapForm: TMapForm;

implementation

uses GameSetup;

{$R *.dfm}

procedure TMapForm.FormCreate(Sender: TObject);
begin
  SystemFont(Font);
  SystemFont(MapList.Header.Font);
  Updater := TMapUpdater.Create(True);
  MapList.NodeDataSize := Sizeof(TMapInfo);
  RefreshFolder;
  GameForm.Setup;
  Updater.Start;
end;

function TMapForm.SelectMap: PMapInfo;
begin
  if ShowModal = mrOK then
    Result := MapList.GetNodeData(SelectedNode)
  else
    Result := nil;
end;

procedure TMapForm.FlagMaps;
var
  Node: PVirtualNode;
  Data: PMapInfo;
begin
  Node := MapList.GetFirst;

  while Node <> nil do
    begin
      Data := MapList.GetNodeData(Node);
      Data.State := msDeleted;
      Node := MapList.GetNext(Node);
    end;
end;

procedure TMapForm.ButtonSelectClick(Sender: TObject);
begin
  if SelectedNode = nil then
    MessageDlg('You must select a map!', mtWarning, [mbOK], 0)
  else
    ModalResult := mrOK;
end;

procedure TMapForm.ClearMaps;
var
  Node, Old: PVirtualNode;
  Data: PMapInfo;
begin
  Node := MapList.GetFirst;

  while Node <> nil do
    begin
      Data := MapList.GetNodeData(Node);

      Old := Node;
      Node := MapList.GetNext(Node);

      case Data.State of
        msDeleted:
          begin
            MapList.DeleteNode(Old);
            Updater.Queue.Remove(Data);
          end;

        msNew:
          begin
            Updater.Queue.Add(Data);
            Data.State := msExisting;
          end;
      end;
    end;
end;

function TMapForm.GetMap(Name: String; FileSize: Integer): PMapInfo;
var
  Node: PVirtualNode;
  Data: PMapInfo;
begin
  Node := MapList.GetFirst;

  while Node <> nil do
    begin
      Data := MapList.GetNodeData(Node);

      if Data.FileName = Name then
      begin
        Data.State := msExisting;
        Exit(Data);
      end;

      Node := MapList.GetNext(Node);
    end;

  Result := MapList.GetNodeData(MapList.AddChild(nil));
  Result.FileName := Name;
  Result.Bitmap := nil;
  Result.State := msNew;
  Result.MapName := Copy(Name, 1, Length(Name) - Length(ExtractFileExt(Name)));

  Result.FileSize := FileSize;
  Result.PlayerCount := -1;

  if FileSize >= 1024 * 1024 then
    Result.FileSizePretty := Format('%f MB', [FileSize / 1024 / 1024])
  else if Result.FileSize >= 1024 then
    Result.FileSizePretty := Format('%f kB', [FileSize / 1024])
  else
    Result.FileSizePretty := IntToStr(FileSize) + ' b';
end;

procedure TMapForm.MapListAfterCellPaint(Sender: TBaseVirtualTree;
  TargetCanvas: TCanvas; Node: PVirtualNode; Column: TColumnIndex;
  CellRect: TRect);
var
  Data: PMapInfo;
  Left, Top: Integer;
begin
 if Column = 0 then
   begin
     Data := Sender.GetNodeData(Node);
     if Data.Bitmap <> nil then
       begin
         Left := CellRect.Left + (CellRect.Right - CellRect.Left) div 2 - Data.Bitmap.Width div 2;
         Top := CellRect.Top + (CellRect.Bottom - CellRect.Top) div 2 - Data.Bitmap.Height div 2;

         BitBlt(TargetCanvas.Handle, Left, Top, Data.Bitmap.Width, Data.Bitmap.Height, Data.Bitmap.Canvas.Handle, 0, 0, SRCCOPY);
       end;
   end;
end;

procedure TMapForm.MapListCompareNodes(Sender: TBaseVirtualTree; Node1,
  Node2: PVirtualNode; Column: TColumnIndex; var Result: Integer);
var
  Data1, Data2: PMapInfo;

begin
  Data1 := Sender.GetNodeData(Node1);
  Data2 := Sender.GetNodeData(Node2);

  case Column of
     1: Result := CompareText(Data1.MapName, Data2.MapName);
     2: Result := Data1.PlayerCount - Data2.PlayerCount;
     3: Result := CompareText(Data1.FileName, Data2.FileName);
     4: Result := Data1.FileSize - Data2.FileSize;
  end;
end;

procedure TMapForm.MapListDblClick(Sender: TObject);
begin
  ButtonSelectClick(nil);
end;

procedure TMapForm.MapListFocusChanged(Sender: TBaseVirtualTree;
  Node: PVirtualNode; Column: TColumnIndex);
begin
  SelectedNode := Node;
end;

procedure TMapForm.MapListFreeNode(Sender: TBaseVirtualTree;
  Node: PVirtualNode);
var Data: PMapInfo;
begin
   Data := MapList.GetNodeData(Node);
   Data.Bitmap.Free;
   Finalize(Data^);
end;

procedure TMapForm.MapListGetText(Sender: TBaseVirtualTree; Node: PVirtualNode;
  Column: TColumnIndex; TextType: TVSTTextType; var CellText: string);
var Data: PMapInfo;
begin
  Data := Sender.GetNodeData(Node);

  case Column of
    1: CellText := Data.MapName;

    2:
      if Data.PlayerCount = -1 then
        CellText := ''
      else
        CellText := IntToStr(Data.PlayerCount);

    3: CellText := Data.FileName;
    4: CellText := Data.FileSizePretty
    else CellText := '';
  end;
end;

procedure TMapForm.MapListHeaderClick(Sender: TVTHeader;
  HitInfo: TVTHeaderHitInfo);
begin
  if HitInfo.Button = mbLeft then
  begin
    with Sender do
      begin
        if SortColumn <> HitInfo.Column then
          SortColumn := HitInfo.Column
        else
          begin
            if SortDirection = sdAscending then
              SortDirection := sdDescending
            else
              SortDirection := sdAscending;
          end;

         TreeView.SortTree(SortColumn, SortDirection, False);
      end;
  end;
end;

procedure TMapForm.RefreshFolder;
var
  SearchRec: TSearchRec;
begin
  try
    Updater.QueueLock.Enter;
    FlagMaps;

    if FindFirst(SearchPath + 'Maps\*.s2ma', faAnyFile, SearchRec) = 0 then
      repeat
        if (SearchRec.Name <> '.') and (SearchRec.Name <> '..') then
          GetMap(SearchRec.Name, SearchRec.Size);
      until FindNext(SearchRec) <> 0;

    FindClose(SearchRec);
  finally
    ClearMaps;
    Updater.QueueLock.Leave;
  end;
end;

procedure TMapForm.FormDestroy(Sender: TObject);
begin
  Updater.Terminate;
  Updater.WaitFor;
  Updater.Free;
end;

procedure TMapForm.FormShow(Sender: TObject);
begin
  RefreshFolder;
end;

end.
