unit GameSetup;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, StdCtrls, MapUpdater, ExtCtrls, IniFiles, Menus;

type
  TPlayer = record
    Ready, Visible: Boolean;
    Index: Integer;
    PlayerType: TComboBox;
    Race: TComboBox;
    Team: TComboBox;
    Color: TColorBox;
    procedure Setup;
    procedure SetVisible(Show: Boolean);
    procedure Save;
  end;

  TGameForm = class(TForm)
    MapBox: TGroupBox;
    ChangeMap: TButton;
    MapImage: TImage;
    MapTitle: TLabel;
    GroupBox3: TGroupBox;
    ModList: TComboBox;
    Label2: TLabel;
    ButtonStart: TButton;
    ButtonCancel: TButton;
    ButtonSettings: TButton;
    PopupMenu: TPopupMenu;
    About: TMenuItem;
    MapDesc: TLabel;
    MapAuthor: TLabel;
    PlayerBox: TGroupBox;
    PlayerType: TLabel;
    PlayerRace: TLabel;
    PlayerTeam: TLabel;
    PlayerColor: TLabel;
    FreeForAll: TCheckBox;
    procedure ChangeMapClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure ButtonCancelClick(Sender: TObject);
    procedure ButtonStartClick(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure AboutClick(Sender: TObject);
    procedure ButtonSettingsClick(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure ColorBox1GetColors(Sender: TCustomColorBox; Items: TStrings);
  private
    { Private declarations }
  public
    SelectedMap: PMapInfo;
    Players: array [0..31] of TPlayer;
    PlayerCount: Integer;
    procedure Setup;
    procedure Save;
    procedure SelectMap(Map: PMapInfo);
    function FindMap(Name: String): PMapInfo;
    procedure UpdateMapInfo(MapHandle: THandle);
    procedure ImportFolder(Path: String; Folder: String; MapHandle: THandle);
  end;

var
  GameForm: TGameForm;
  SearchPath: String;
  Settings: TIniFile;

procedure SystemFont(Font: TFont);

implementation

uses MapSelector, pMpqAPI, VirtualTrees, LaunchSettings, StreamUtils;

{$R *.dfm}

procedure SystemFont(Font: TFont);
var
  LogFont: TLogFont;
begin
  if SystemParametersInfo(SPI_GETICONTITLELOGFONT, SizeOf(TLogFont), @LogFont, 0) then
    begin
      Font.Height := LogFont.lfHeight;
      Font.Orientation := LogFont.lfOrientation;
      Font.Charset := TFontCharset(LogFont.lfCharSet);
      Font.Name := PChar(@LogFont.lfFaceName);

      Font.Style := [];

      if LogFont.lfWeight >= FW_BOLD then
        Font.Style := Font.Style + [fsBold];

      if LogFont.lfItalic = 1 then
        Font.Style := Font.Style + [fsItalic];

      if LogFont.lfUnderline = 1 then
        Font.Style := Font.Style + [fsUnderline];

      if LogFont.lfStrikeOut = 1 then
        Font.Style := Font.Style + [fsStrikeOut];

      case LogFont.lfPitchAndFamily and $F of
        VARIABLE_PITCH: Font.Pitch := fpVariable;
        FIXED_PITCH: Font.Pitch := fpFixed;
        else Font.Pitch := fpDefault;
      end;
    end;
end;

procedure TPlayer.Save;
begin
  if not Ready then
    Exit;

  Settings.WriteInteger('Player' + IntToStr(Index), 'Type', PlayerType.ItemIndex);
  Settings.WriteInteger('Player' + IntToStr(Index), 'Race', Race.ItemIndex);
  Settings.WriteInteger('Player' + IntToStr(Index), 'Team', Team.ItemIndex);
end;

procedure TPlayer.Setup;
var
  Parent: TWinControl;
  Top: Integer;
begin
  Parent := GameForm.PlayerBox;
  PlayerType := TComboBox.Create(Parent);
  Race := TComboBox.Create(Parent);
  Team := TComboBox.Create(Parent);
  Color := TColorBox.Create(Parent);

  PlayerType.Visible := False;
  Race.Visible := False;
  Team.Visible := False;
  Color.Visible := False;

  PlayerType.Parent := Parent;
  Race.Parent := Parent;
  Team.Parent := Parent;
  Color.Parent := Parent;

  Top := GameForm.PlayerType.Top + GameForm.PlayerType.Height + 7 + (GameForm.ModList.Height + 5) * Index;

  PlayerType.Left := GameForm.PlayerType.Left;
  PlayerType.Top := Top;
  PlayerType.Width := GameForm.PlayerRace.Left - PlayerType.Left - 5;
  PlayerType.Items.Text := 'Player'#13#10'Computer'#13#10'Disabled';
  PlayerType.Style := csDropDownList;

  if Index = 0 then
    PlayerType.ItemIndex := Settings.ReadInteger('Player' + IntToStr(Index), 'Type', 0)
  else
    PlayerType.ItemIndex := Settings.ReadInteger('Player' + IntToStr(Index), 'Type', 1);

  Race.Left := GameForm.PlayerRace.Left;
  Race.Top := Top;
  Race.Width := GameForm.PlayerTeam.Left - Race.Left - 5;
  Race.Items.Text := 'Random'#13#10'Terran'#13#10'Protoss'#13#10'Zerg';
  Race.Enabled := False;
  Race.Style := csDropDownList;
  Race.ItemIndex := Settings.ReadInteger('Player' + IntToStr(Index), 'Race', 0);

  Team.Left := GameForm.PlayerTeam.Left;
  Team.Top := Top;
  Team.Width := GameForm.PlayerColor.Left - Team.Left - 5;
  Team.Items.Text := 'Team 1'#13#10'Team 2'#13#10'Team 3'#13#10'Team 4';
  Team.Enabled := False;
  Team.Style := csDropDownList;

  if Index = 0 then
    Team.ItemIndex := Settings.ReadInteger('Player' + IntToStr(Index), 'Team', 0)
  else
    Team.ItemIndex := Settings.ReadInteger('Player' + IntToStr(Index), 'Team', 1);

  Color.Left := GameForm.PlayerColor.Left;
  Color.Top := Top;
  Color.Width := GameForm.PlayerBox.ClientWidth - Color.Left - 10;
  Color.Style := [cbPrettyNames, cbCustomColors];
  Color.Enabled := False;


  Ready := True;
end;

procedure TPlayer.SetVisible(Show: Boolean);
begin
  if Visible = Show then
    Exit;

  if not Ready then
    Setup;

  Visible := Show;

  PlayerType.Visible := Show;
  Race.Visible := Show;
  Team.Visible := Show;
  Color.Visible := Show;
end;

procedure TGameForm.AboutClick(Sender: TObject);
begin
  MessageDlg('Starcraft II Launcher 2.0'#13#10' Zoxc 2010'#13#10#13#10'Thanks to:'#13#10' - ShadowFlare for SFmpq'#13#10' - Team Lazytown for their launcher'#13#10' - Team StarCrack for the included AI'#13#10' - Blizzard for StarCraft II', mtInformation, [mbOK], 0)
end;

procedure TGameForm.ButtonCancelClick(Sender: TObject);
begin
  Close;
end;

procedure TGameForm.ImportFolder(Path: String; Folder: String; MapHandle: THandle);
var
  SearchRec: TSearchRec;
begin
  if FindFirst(Path + Folder + '*.*', faAnyFile, SearchRec) = 0 then
    begin
      repeat
        if (SearchRec.Name <> '.') and (SearchRec.Name <> '..') then
          begin
            if (SearchRec.Attr and faDirectory) <> 0 then
              ImportFolder(Path, Folder + SearchRec.Name + '\', MapHandle)
            else
              begin
                if not MpqAddFileToArchive(MapHandle, PAnsiChar(AnsiString(Path + Folder + SearchRec.Name)), PAnsiChar(AnsiString(Folder + SearchRec.Name)), MAFA_REPLACE_EXISTING) then
                  raise Exception.Create('Unable to import file ''' +  Folder + SearchRec.Name + '''.');
              end;
         end;
      until FindNext(SearchRec) <> 0;
   end;
  FindClose(SearchRec);
end;

procedure TGameForm.UpdateMapInfo(MapHandle: THandle);
var
  MemoryStream: TMemoryStream;
  PlayerType: Byte;
  i: Integer;
begin
  MemoryStream := ReadFile('MapInfo');
  try
    MemoryStream.Position := 24; // Skip fields
    SkipNullString(MemoryStream); // Tileset
    SkipNullString(MemoryStream); // Title
    MemoryStream.Seek(24, soFromCurrent); // Skip fields
    SkipNullString(MemoryStream); // Loading image
    MemoryStream.Seek(33, soFromCurrent); // Skip fields

    MemoryStream.Seek(12, soFromCurrent); // Skip first player
    SkipNullString(MemoryStream); // Skip string in first player
    MemoryStream.Seek(10, soFromCurrent); // Skip rest of first player

    for i := 0 to PlayerCount - 1 do
      begin
        PlayerType := 0;

          case Players[i].PlayerType.ItemIndex of
            0: PlayerType := 1;
            1: PlayerType := 2;
            2: PlayerType := 0;
          end;

        MemoryStream.Seek(4, soFromCurrent); // Skip fields
        MemoryStream.Write(PlayerType, 1); // Change player Type

        MemoryStream.Seek(7, soFromCurrent); // Skip fields

        SkipNullString(MemoryStream); // Skip string

        MemoryStream.Seek(10, soFromCurrent); // Skip fields
      end;

    if not MpqAddFileFromBuffer(MapHandle, MemoryStream.Memory, MemoryStream.Size - 1, 'MapInfo', MAFA_REPLACE_EXISTING) then
      raise Exception.Create('Unable to replace MapInfo.');
  finally
    MemoryStream.Free;
  end;
end;

procedure TGameForm.ButtonSettingsClick(Sender: TObject);
begin
  SettingsForm.Open;
end;

procedure TGameForm.ButtonStartClick(Sender: TObject);
var
  MapHandle: THandle;
  OutputMap, Parameters: String;
  StartupInfo: TStartupInfo;
  ProcessInfo: TProcessInformation;
  i, YouCount: Integer;
  HasPlayer: Boolean;
begin

  if SelectedMap = nil then
    begin
      MessageDlg('You can''t start without a map!', mtWarning, [mbOK], 0);
      Exit;
    end;

  HasPlayer := False;
  YouCount := 0;

  for i := Low(Players) to High(Players) do
    begin
      if not HasPlayer then
        HasPlayer := Players[i].Visible and ((Players[i].PlayerType.ItemIndex = 0) or (Players[i].PlayerType.ItemIndex = 1));

      if Players[i].Visible and (Players[i].PlayerType.ItemIndex = 0) then
        Inc(YouCount);
    end;

  if not HasPlayer then
    begin
      MessageDlg('You can''t start without any players!', mtWarning, [mbOK], 0);
      Exit;
    end;

  if YouCount > 1 then
    begin
      MessageDlg('You can''t play multiple players!', mtWarning, [mbOK], 0);
      Exit;
    end;

  Save;

  OutputMap := SearchPath + 'Files\Map.s2ma';

  CopyFile(PChar(SearchPath + 'Maps\' + SelectedMap.FileName), PChar(OutputMap), False);

  MapHandle := MpqOpenArchiveForUpdate(PAnsiChar(AnsiString(OutputMap)), MOAU_OPEN_EXISTING, 0);

  if MapHandle = 0 then
    raise Exception.Create('Unable to open map!');

  try
    ImportFolder(SearchPath + 'Base\', '', MapHandle);

    if ModList.ItemIndex <> 0 then
      ImportFolder(SearchPath + 'Mods\' + ModList.Text + '\', '', MapHandle);

    UpdateMapInfo(MapHandle);
  finally
    MpqCloseUpdatedArchive(MapHandle, 0);
  end;

  Parameters := '';

  if Settings.ReadBool('SC2Launcher', 'TriggerDebug', False) then
    Parameters := Parameters + ' -trigdebug=1';

  if Settings.ReadBool('SC2Launcher', 'ShowFPS', False) then
    Parameters := Parameters + ' -showfps=1';

  if Settings.ReadBool('SC2Launcher', 'CaptureMouse', False) then
    Parameters := Parameters + ' -confinecursor=2';

  if Settings.ReadBool('SC2Launcher', 'DisableAltTab', False) then
    Parameters := Parameters + ' -alttabdisabled=1';

  if Settings.ReadBool('SC2Launcher', 'DisableWinKey', False) then
    Parameters := Parameters + ' -winkeydisabled=1';

  if Settings.ReadBool('SC2Launcher', 'DisableTextures', False) then
    Parameters := Parameters + ' -noTextureLoading=1';

  if Parameters <> '' then
    Parameters := ' -- ' + Parameters;

  ZeroMemory(@StartupInfo, SizeOf(TStartupInfo));
  ZeroMemory(@ProcessInfo, SizeOf(TProcessInformation));

  StartupInfo.cb := SizeOf(TStartupInfo);

  if not CreateProcess(nil, PChar('"' + SearchPath + 'Files\lazylaunch2.exe" "' + OutputMap + '"' + Parameters), nil, nil, False, 0, nil, nil, StartupInfo, ProcessInfo) then
    RaiseLastOSError;

  CloseHandle(ProcessInfo.hThread);
  CloseHandle(ProcessInfo.hProcess);
end;

procedure TGameForm.ChangeMapClick(Sender: TObject);
var
  Map: PMapInfo;
begin
  Map := MapForm.SelectMap;

  if Map <> nil then
    SelectMap(Map);
end;

procedure TGameForm.ColorBox1GetColors(Sender: TCustomColorBox;
  Items: TStrings);
begin
  Items.Text := 'Black'#13#10'Red';
end;

procedure TGameForm.FormClose(Sender: TObject; var Action: TCloseAction);
begin
  Save;
end;

procedure TGameForm.FormCreate(Sender: TObject);
var
  SearchRec: TSearchRec;
  i: Integer;
begin
  for i := Low(Players) to High(Players) do
    Players[i].Index := i;

  SystemFont(Font);
  MapTitle.Font.Style := MapTitle.Font.Style + [fsBold];
  MapAuthor.Font.Color := clGrayText;
  SearchPath := IncludeTrailingPathDelimiter(ExtractFilePath(Application.ExeName));

  Settings := TIniFile.Create(ChangeFileExt(Application.ExeName, '.ini'));

  if FindFirst(SearchPath + 'Mods\*.*', faAnyFile, SearchRec) = 0 then
    begin
      repeat
        if (SearchRec.Name <> '.') and (SearchRec.Name <> '..') and ((SearchRec.Attr and faDirectory) <> 0) then
          ModList.Items.Add(SearchRec.Name);
      until FindNext(SearchRec) <> 0;
   end;

  FindClose(SearchRec);

  if ModList.Items.Count > 0 then
    ModList.ItemIndex := 0
  else
    ModList.Enabled := False;
end;

procedure TGameForm.FormDestroy(Sender: TObject);
begin
  Settings.Free;
end;

procedure TGameForm.SelectMap(Map: PMapInfo);
var i, NewPlayerCount: Integer;
begin
  if Map <> nil then
    begin
      MapImage.Visible := True;
      MapImage.Picture.Bitmap.Assign(Map.Bitmap);

      MapTitle.Top := MapImage.Top + MapImage.Height + 7;

      if Map.PlayerCount <> -1 then
        MapTitle.Caption := Map.MapName + ' (' + IntToStr(Map.PlayerCount) + ')'
      else
        MapTitle.Caption := Map.MapName;

      MapAuthor.Top := MapTitle.Top + MapTitle.Height;
      MapAuthor.Caption := Map.MapAuthor;

      MapDesc.Top := MapAuthor.Top + MapAuthor.Height + 7;
      MapDesc.Caption := Map.MapDesc;
      MapDesc.Height := ChangeMap.Top - 10 - MapDesc.Top;

      if Map.PlayerCount = -1 then
        NewPlayerCount := 0
      else
        NewPlayerCount := Map.PlayerCount;

      if PlayerCount > NewPlayerCount then
        begin
          for i := NewPlayerCount to PlayerCount - 1 do
            if i < 32 then
              Players[i].SetVisible(False);
        end
      else
        begin
          for i := PlayerCount to NewPlayerCount - 1 do
            if i < 32 then
              Players[i].SetVisible(True);
        end;

      PlayerCount := NewPlayerCount;
    end
  else
    begin
      for i := 0 to PlayerCount - 1 do
        if i < 32 then
          Players[i].SetVisible(False);

      PlayerCount := 0;

      MapDesc.Caption := 'No map selected.';
      MapDesc.Top := MapImage.Top;
      MapAuthor.Caption := '';
      MapTitle.Caption := '';
      MapImage.Visible := False;
    end;

  SelectedMap := Map;
end;

function TGameForm.FindMap(Name: String): PMapInfo;
var
  Node: PVirtualNode;
  Data: PMapInfo;
begin
  Node := MapForm.MapList.GetFirst;

  while Node <> nil do
    begin
      Data := MapForm.MapList.GetNodeData(Node);
      if Data.FileName = Name then
        Exit(Data);

      Node := MapForm.MapList.GetNext(Node);
    end;

  Result := nil;
end;

procedure TGameForm.Save;
var
  i: Integer;
begin
  if SelectedMap = nil then
    Settings.WriteString('SC2Launcher', 'Map', '')
  else
    Settings.WriteString('SC2Launcher', 'Map', SelectedMap.FileName);

  Settings.WriteString('SC2Launcher', 'Mod', ModList.Text);

  for i := Low(Players) to High(Players) do
    Players[i].Save;
end;

procedure TGameForm.Setup;
var
  Map: String;
  ModName: String;
  Data: PMapInfo;
  Index: Integer;
begin
  Map := Settings.ReadString('SC2Launcher', 'Map', '');

  if Map <> '' then
    Data := FindMap(Map)
  else
    Data := nil;

  if (Data = nil) and (MapForm.MapList.GetFirst <> nil) then
    Data := MapForm.MapList.GetNodeData(MapForm.MapList.GetFirst);

  SelectMap(Data);

  ModName := Settings.ReadString('SC2Launcher', 'Mod', '');

  Index := ModList.Items.IndexOf(ModName);
  if Index <> -1 then
    ModList.ItemIndex := Index;
end;

end.
